/***************************************************************************** 
 *                         API HOOKER Library                                *
 *                                                                           *
 * This library allows you to hook API Functions                             *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *    Copyright (c)2001           Toby Opferman                              *
 *****************************************************************************/

#include <windows.h>
#include "apihooker.h"

/***************************************************************************** 
 * Internal Data Structure                                                   *
 *****************************************************************************/
#define JUMP_CODE_SIZE  6
#define JUMP_OPCODE     ((WORD)0x25FF)

typedef struct {

  BYTE FunctionReplacedBytes[JUMP_CODE_SIZE];
  PVOID FunctionLocation;
  PVOID HookLocation;

} API_HOOK_INTERNAL, *PAPI_HOOK_INTERNAL;


/*******************************************************************************
 * DllMain                                                                     *
 *                                                                             *
 * DESCRIPTION: DLL Entry Point                                                *
 *                                                                             *
 * INPUT                                                                       *
 *   Standard Win32 DLL Entry                                                  *
 *                                                                             *
 * OUTPUT                                                                      * 
 *   TRUE = Initialized, FALSE = Failure                                       *
 *                                                                             *
 *******************************************************************************/
BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD dwReason, LPVOID lpvReserved)
{
    return TRUE;
}



/*******************************************************************************
 * ApiHooker_ReplaceAPIWithHook                                                *
 *                                                                             *
 * DESCRIPTION: Replace API With Hook Function by replacing the                *
 *              first few bytes at the beginning of the function with          *
 *              a jump to the new function.                                    *
 *                                                                             *
 * INPUT                                                                       *
 *   Location of Function to be Hooked, Hook Function Location                 *
 *                                                                             *
 * OUTPUT                                                                      * 
 *   Handle to a Hook API                                                      *
 *                                                                             *
 *******************************************************************************/
HAPIHOOK ApiHooker_ReplaceAPIWithHook(PVOID FunctionLocation, PVOID HookLocation)
{
    PAPI_HOOK_INTERNAL pApiHook = INVALID_APIHOOK_VALUE;
    DWORD OldProtectionValue, ReadWriteProtectionValue;
    WORD  *pOpcode          = (WORD *)FunctionLocation;
    DWORD *pFunctionAddress = (DWORD *)((WORD *)FunctionLocation + 1);

    if(pApiHook = LocalAlloc(LMEM_ZEROINIT, sizeof(API_HOOK_INTERNAL)))
    {
        pApiHook->FunctionLocation = FunctionLocation;
        pApiHook->HookLocation     = HookLocation; 

        VirtualProtect(FunctionLocation, JUMP_CODE_SIZE, PAGE_READWRITE, &OldProtectionValue);

        memcpy(pApiHook->FunctionReplacedBytes, FunctionLocation, JUMP_CODE_SIZE);

        *pOpcode          = JUMP_OPCODE;
        *pFunctionAddress = (DWORD)&pApiHook->HookLocation;
 
        VirtualProtect(FunctionLocation, JUMP_CODE_SIZE, OldProtectionValue, &ReadWriteProtectionValue);
    }

    return (HAPIHOOK)pApiHook;
}


/*******************************************************************************
 * ApiHooker_FreeHook                                                          *
 *                                                                             *
 * DESCRIPTION: Restore API by replacing back the 5 bytes at the               *
 *              top of the function.                                           *
 *                                                                             *
 * INPUT                                                                       *
 *   Handle to an API Hook                                                     *
 *                                                                             *
 * OUTPUT                                                                      * 
 *   Nothing                                                                   *
 *                                                                             *
 *******************************************************************************/
void ApiHooker_FreeHook(HAPIHOOK hApiHook)
{
    PAPI_HOOK_INTERNAL pApiHook = (PAPI_HOOK_INTERNAL)hApiHook;
    DWORD OldProtectionValue, ReadWriteProtectionValue;

    VirtualProtect(pApiHook->FunctionLocation, JUMP_CODE_SIZE, PAGE_READWRITE, &OldProtectionValue);

    memcpy(pApiHook->FunctionLocation, pApiHook->FunctionReplacedBytes, JUMP_CODE_SIZE);
 
    VirtualProtect(pApiHook->FunctionLocation, JUMP_CODE_SIZE, OldProtectionValue, &ReadWriteProtectionValue);

    LocalFree(pApiHook);
}





